// video module for the DE2 board
// Dec 10 2013
// by Fred Aulich
// capture frame from video in and save in low and high memory
// last update Nov 8 2013 -- changed sync for video in 
// last update Nov 19 2013 -- clean up code update monitor and camera module
// last update Dec 20 2013 -- added I2C programming code 
// updated for DE1-SoC add on chip memory Dec 2015

module video (
// video out chip
videorgb		, // 8 bit video value 3 red, 3 blue, 2 green
clk			, // 50 Mhz clock 
vid_clk		, // 25 Mhz clock to video chip
clk_27		, // 27 Mhz clock for video in chip
clken			, // clock enable for 27 Mhz clock
vid_hs		, // horizontal sync camera capture
vid_vs		, // vertical sync camera capture
video_in		, // 8 bit video input from camera
sda			, // serial data bus
scl			, // clock line serial data bus
vsync			, // video vertical sync
hsync			, // video horizontal sync
vid_blank	, // video blank 
swt			, // switch inputs
led			, // LED outputs
gpio			 // general purpose registers

);
// signal directions

output[7:0]		videorgb;
output[7:0]		led;
output[15:0] 	gpio;
output			clken;
output 			scl;

input			clk;
input 		clk_27;
input[7:0] 	video_in;
input[7:0]	swt;

inout		sda;
inout		vid_hs;
inout		vid_vs;
inout		vid_clk;
inout 	vsync;
inout 	hsync; 
inout		vid_blank;

// registered pin

reg[10:0]	clkcount; // clock divider

//////////////////////////////////////
///// monitor module					/////
//////////////////////////////////////

wire[7:0]	videorgb;
wire[7:0] 	video_mot;
wire[15:0] 	address_mot;
wire 			hsync;
wire			vsync;
wire			vid_blank;
wire 			read_hl;
wire 			read_hh;
wire 			read;
wire 			read_lh;
wire 			read_ll;

//////////////////////////////////////
/// camera  module					 ////
//////////////////////////////////////

wire[15:0] 	address_cam;
wire 			vid_ldl;
wire 			vid_ldh;
wire 			vid_udl;
wire 			vid_udh;
wire			we_cam;
wire			clk_27;
wire 			oddeven;
wire			frame;


////////////////////////////////////////////
//  memory module from library			/////
////////////////////////////////////////////

wire[15:0]	data_caml;  // data from camera
wire[15:0]	data_camh;
wire[31:0]  data_in;
wire[15:0]  data_motl;
wire[15:0]  data_moth;
wire[31:0]  data_q;
wire[3:0]	vid_en;
wire[15:0]	rdaddress;
wire[15:0]	wraddress;

/////////////////////////////////////////////////////////////////////////////////
//// modules for video in, video out, I2C programmer and on chip memory module //
/////////////////////////////////////////////////////////////////////////////////


camera  u0(

		.clk_27(clk_27),		   // 27 Mhz clock
		.vid_vs(vid_vs),			// vertical sync
		.vid_hs(vid_hs),			// horizontal sync
		.video_in(video_in),		// video in data from camera
		.address_cam(address_cam),
		.data_caml(data_caml),
		.data_camh(data_camh),
		.frame(frame),
		.vid_ldl(vid_ldl),
		.vid_ldh(vid_ldh),
		.vid_udl(vid_udl),
		.vid_udh(vid_udh),
		.we_cam(we_cam)

);

monitor u1 (			
			.vid_clk(vid_clk),			// 25 Mhz clock to video chip
			.clk(clk),						// 50 Mhz clock
			.vsync(vsync), 				// video vertical sync
			.hsync(hsync), 				// video horizontal sync
			.vid_blank(vid_blank), 		// video blank 
			.data_motl(data_motl),   	// data from memory low word
			.data_moth(data_moth),		// data from memory high word
			.read_lowl(read_ll),	 		// odd byte low		
			.read_highl(read_lh), 		// odd byte high
			.read_lowh(read_hl),  		// even byte low
			.read_highh(read_hh), 		// even bytee high
			.read(read),
			.address_mot(address_mot),
			.oddeven(oddeven),
			.video_mot(video_mot)
);

I2C_programmer u2(
 
			.RESET(swt[1]),			//  27 MHz clock enable 
			.i2c_clk(clk_27),		//  27 Mhz clk from DE1-SoC
			.I2C_SCLK(scl),		// I2C clock 40K
			.TRN_END(TRN_END),
			.ACK(ACK),
			.ACK_enable(ACK_enable),
			.I2C_SDATA(sda)		// bi directional serial data 
 );
 
 ////////////////////////////////////
 // on board memory modules      ////
 ////////////////////////////////////
 
ram_16 u3 (

			.clock(clk),
			.wraddress(wraddress),
			.rdaddress(rdaddress),
			.data(data_in),
			.byteena_a(vid_en),
			.q(data_q),
			.rden(read),
			.wren(!read)
);

////////////////////////////////////////////
/// variables for loading I2C programmer ///
////////////////////////////////////////////
 
 wire scl; 
 wire sda; 
 
 wire ACK ;
 wire ACK_enable;
 wire [23:0] data_23;
 wire TRN_END;
 
////////////////////////////////////
/// general clock divider        ///
////////////////////////////////////

 
 always @ (posedge clk )

begin 
	if (!swt[0]) 
		begin		
		clkcount = 0;
		end
		
		else
		begin
		clkcount <= clkcount + 1;
		end
end

	
////////////////////////////////
// fixed output values      ////
////////////////////////////////

////////////////////////////////
///  memory module          ////
////////////////////////////////
/// 16 address lines        ////
/// 32 data lines           ////
/// 4 byte enables          ////
////////////////////////////////
 

	 assign wraddress = address_cam; 	// address counter in 
	 assign rdaddress = address_mot;    // address counter out
	 
	 assign data_in[31:16] = data_camh; // upper 16 bits data in
	 assign data_in[15:0] = data_caml;  // lower 16 bits data in 
	 
	 assign data_motl = data_q[15:0]; // lower 16 bits data out 
	 assign data_moth = data_q[31:16];// upper 16 bits data out
	 
	 assign vid_en[0] = vid_ldl;  // video in byte enable
	 assign vid_en[1] = vid_ldh;	// video in byte enable
	 assign vid_en[2] = vid_udl;  // video in byte enable
	 assign vid_en[3] = vid_udh;  // video in byte enable

	 assign vid_clk = clkcount[0]; // 25 Mhz clock	
	 assign clken = swt[0];        // camera reset
	 assign led = swt;
	 assign videorgb = video_mot;  // 8 bit video out display register

///////////////////////////////////////////////	
// outputs to debug circuit gpio(0)	      /////
///////////////////////////////////////////////
	
	assign gpio[0] = vid_clk;
	assign gpio[1] = read;
	assign gpio[2] = we_cam;
	assign gpio[3] = oddeven;
	assign gpio[4] = vid_ldl;
	assign gpio[5] = vid_ldh;
	assign gpio[6] = vid_udl;
	assign gpio[7] = vid_udh;

	assign gpio[15:8] = address_cam[7:0];


endmodule	
